' Using EMS in QuickBasic: Part 2 of 3
' Source Code
'
' By Jon Petrosky (Plasma)
' www.phatcode.net
'
' [Remember to start QB with the /L switch to enable interrupts]

DEFINT A-Z
'$DYNAMIC
'$INCLUDE: 'QB.BI'

DIM SHARED Regs AS RegTypeX
DIM SHARED EMS.Error            'Holds the error code of the last operation

DECLARE FUNCTION EMS.ErrorMsg$ ()
DECLARE FUNCTION EMS.Init ()
DECLARE FUNCTION EMS.Version$ ()
DECLARE FUNCTION EMS.PageFrame ()
DECLARE FUNCTION EMS.FreeHandles ()
DECLARE FUNCTION EMS.FreePages ()
DECLARE FUNCTION EMS.TotalPages ()
DECLARE FUNCTION EMS.AllocPages (NumPages)
DECLARE SUB EMS.DeallocPages (Handle)
DECLARE SUB EMS.MapPage (Physical, Logical, Handle)
DECLARE SUB EMS.MapXPages (PhysicalStart, LogicalStart, NumPages, Handle)
DECLARE SUB EMS.CopyMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)
DECLARE SUB EMS.ExchMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)

DECLARE SUB SB.GetConfig ()
DECLARE FUNCTION SB.Init ()
DECLARE FUNCTION SB.LoadSound (Filename$, Slot, Priority, Handle)
DECLARE SUB SB.WriteDSP (Byte)
DECLARE FUNCTION SB.InUse ()
DECLARE SUB SB.PlaySound (Slot, Handle)
DECLARE SUB SB.SpeakerOn ()
DECLARE SUB SB.SpeakerOff ()
DECLARE SUB SB.Pause ()
DECLARE SUB SB.Resume ()

DIM SHARED SB.BaseAddr  'Sound Blaster base address
DIM SHARED SB.DMAchan   'Sound Blaster DMA channel
DIM SHARED SB.DMApage   'DMA page register (set by SB.Init)
DIM SHARED SB.DMAadd    'DMA address register (set by SB.Init)
DIM SHARED SB.DMAlen    'DMA length register (set by SB.Init)
DIM SHARED SB.Sound     'Currently playing sound slot (set by SB.PlaySound)

CONST SB.MaxSounds = 9  'Maximum number of sound slots. This can be changed,
			'  just remember that 64K is required for each slot.

'Holds the frequency, length, and priority of each sound slot.
DIM SHARED SB.SlotInfo(1 TO SB.MaxSounds, 1 TO 3) AS LONG

COLOR 7
CLS
IF NOT EMS.Init THEN
  PRINT "No EMM detected."
  END
END IF

SB.GetConfig
IF SB.BaseAddr = 0 OR SB.DMAchan = 0 THEN
  PRINT "Sound Blaster settings not found. Please enter them manually."
  PRINT
  INPUT "Base address (usually 220): ", BaseAddr$
  SB.BaseAddr = VAL("&H" + BaseAddr$)
  INPUT "DMA channel (usually 1): ", SB.DMAchan
  PRINT
END IF

IF NOT SB.Init THEN
  PRINT "No Sound Blaster detected."
  END
END IF

CLS
COLOR 14, 1
PRINT SPACE$(22); "Using EMS in QuickBasic: Part 2 of 3"; SPACE$(22)
COLOR 15, 0
PRINT STRING$(31, 196); " EMS Information "; STRING$(32, 196)
COLOR 7
PRINT "EMM Version: "; EMS.Version$

IF EMS.Version$ < "4.0" THEN
  PRINT
  PRINT "EMM 4.0 or later must be present to use some of the EMS functions."
  END
END IF

PRINT "Page frame at: "; HEX$(EMS.PageFrame); "h"
PRINT "Free handles:"; EMS.FreeHandles

IF EMS.FreeHandles = 0 THEN
  PRINT
  PRINT "You need at least one free handle to run this demo."
  END
END IF

PRINT "Total EMS:"; EMS.TotalPages; "pages /"; EMS.TotalPages * 16&; "KB /"; EMS.TotalPages \ 64; "MB"
PRINT "Free EMS:"; EMS.FreePages; "pages /"; EMS.FreePages * 16&; "KB /"; EMS.FreePages \ 64; "MB"

IF EMS.FreePages < 36 THEN
  PRINT
  PRINT "You need at least 36 pages (576 KB) free EMS to run this demo."
  END
END IF

PRINT
COLOR 15
PRINT STRING$(26, 196); " Sound Blaster Information "; STRING$(27, 196)
COLOR 7
PRINT "Base Address: "; HEX$(SB.BaseAddr); "h"
PRINT "DMA Channel:"; SB.DMAchan
PRINT
COLOR 15
PRINT STRING$(30, 196); " Setting Up Sounds "; STRING$(31, 196)
COLOR 7
PRINT "Allocating 36 pages (576 KB) of EMS...";

Handle = EMS.AllocPages(64)
IF EMS.Error THEN
  PRINT "error!"
  PRINT EMS.ErrorMsg$
  END
ELSE
  PRINT "ok! (Using handle "; LTRIM$(STR$(Handle)); ")"
END IF

FOR Sfx = 1 TO 9

  PRINT "Loading SFX"; LTRIM$(STR$(Sfx)); ".WAV into slot "; LTRIM$(STR$(Sfx)); "...";

  SELECT CASE Sfx
    CASE 1, 3, 9
      Priority = 1
    CASE 4, 7, 8
      Priority = 2
    CASE 5, 6
      Priority = 3
    CASE 2
      Priority = 4
  END SELECT

  IF NOT SB.LoadSound("SFX" + LTRIM$(STR$(Sfx)) + ".WAV", Sfx, Priority, Handle) THEN
    PRINT "error!"
    PRINT "couldn't load wave file"
    END
  ELSE
    PRINT "ok! ("; LTRIM$(STR$(SB.SlotInfo(Sfx, 1))); " Hz,"; SB.SlotInfo(Sfx, 2); "bytes, #"; LTRIM$(STR$(SB.SlotInfo(Sfx, 3))); " priority)"
  END IF

NEXT

LOCATE 25, 28
COLOR 31
PRINT "Press any key to proceed";

KeyPress$ = INPUT$(1)
COLOR 7
CLS
COLOR 14, 1
PRINT SPACE$(22); "Using EMS in QuickBasic: Part 2 of 3"; SPACE$(22)
COLOR 15, 0
PRINT STRING$(34, 196); " Sound Test "; STRING$(34, 196)
COLOR 7
PRINT
PRINT "Here you can test the sound playback routines. In this demo, only 9 sounds"
PRINT "have been loaded into EMS. However, in your own programs you may load as many"
PRINT "sounds as you want! The only limitation is that you must have enough free EMS."
PRINT "(And I don't think that should be a problem... ^_^)"
PRINT
COLOR 15
PRINT "Press 1 through 9 to hear different sounds."
PRINT
PRINT "Press P to pause/stop sound playback and R to resume a paused sound."
PRINT
PRINT "Press ESC to end the demo."
PRINT
COLOR 7
PRINT "For example, press 2 to hear a long drone. After the sound has started, press"
PRINT "P and the sound will pause. You can then play a different sound or resume the"
PRINT "same sound by pressing R."
PRINT
PRINT "You will also notice that some sounds have a higher priority than other sounds."
PRINT "For instance, sound 9 has a #1 priority while sound 4 has a #2 priority. This"
PRINT "means that sound 9 will interrupt sound 4 if it is playing. However, if sound 9"
PRINT "is playing, sound 4 will not interrupt it (because sound 9 has a lower priority)"

SB.SpeakerOn

COLOR 15
DotPos = 1
DotDir = 1
DO
  KeyPress$ = UCASE$(INKEY$)
  SELECT CASE KeyPress$
    CASE "1" TO "9"
      SB.PlaySound VAL(KeyPress$), Handle
    CASE "P"
      SB.Pause
    CASE "R"
      SB.Resume
  END SELECT

  IF SB.InUse THEN
    Sound$ = LTRIM$(STR$(SB.Sound))
    Freq$ = LTRIM$(STR$(SB.SlotInfo(SB.Sound, 1)))
    Size$ = LTRIM$(STR$(SB.SlotInfo(SB.Sound, 2)))
    Priority$ = LTRIM$(STR$(SB.SlotInfo(SB.Sound, 3)))
    LOCATE 25, 1
    COLOR 15
    PRINT "Playing sound #"; Sound$; " ("; Freq$; " Hz, "; Size$; " bytes, #"; Priority$; " priority)"; SPACE$(12);
    DotCol = 10
  ELSE
    LOCATE 25, 1
    COLOR 15
    PRINT "No sound is playing"; SPACE$(40);
    DotCol = 12
  END IF

  WAIT &H3DA, 8, 8
  WAIT &H3DA, 8

  LOCATE 24, DotPos
  COLOR DotCol
  PRINT CHR$(254);
  IF DotPos > 1 THEN
    LOCATE 24, DotPos - 1
   PRINT " ";
  END IF
  IF DotPos < 80 THEN
    LOCATE 24, DotPos + 1
    PRINT " ";
  END IF
  IF DotDir = 1 THEN
    DotPos = DotPos + 1
    IF DotPos = 81 THEN
      DotPos = 80
      DotDir = 2
    END IF
  ELSEIF DotDir = 2 THEN
    DotPos = DotPos - 1
    IF DotPos = 0 THEN
      DotPos = 1
      DotDir = 1
    END IF
  END IF

LOOP UNTIL KeyPress$ = CHR$(27)

SB.SpeakerOff

COLOR 7
CLS
COLOR 14, 1
PRINT SPACE$(22); "Using EMS in QuickBasic: Part 2 of 3"; SPACE$(22)
COLOR 15, 0
PRINT STRING$(33, 196); " Ending Demo "; STRING$(34, 196)
COLOR 7
PRINT
PRINT "Deallocating 36 pages...";

EMS.DeallocPages (Handle)
IF EMS.Error THEN
  PRINT "error!"
  PRINT EMS.ErrorMsg$
  END
ELSE
  PRINT "ok!"
END IF

END

FUNCTION EMS.AllocPages (NumPages)

  'Allocates the number of pages in [NumPages] and
  'returns the EMS handle the memory is allocated to.

  Regs.ax = &H4300                           'Allocate [NumPages] pages of EMS
  Regs.bx = NumPages
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

  EMS.AllocPages = Regs.dx                   'Return the handle

END FUNCTION

SUB EMS.CopyMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)

  'Copies memory from EMS or base memory to EMS or base memory, where:
  '
  'Length&    = Length of memory to copy in bytes
  'SrcHandle  = EMS handle of source memory (use 0 if source is base memory)
  'SrcSegment = Segment of source memory (or page number if source is EMS)
  'SrcOffset  = Offset of source memory
  'DstHandle  = EMS handle of destination memory (use 0 if destination is base memory)
  'DstSegment = Segment of destination memory (or page number if destination is EMS)
  'DstOffset  = Offset of destination memory

  'Determine the source and destination memory types by checking the handles
  IF SrcHandle = 0 THEN SrcType$ = CHR$(0) ELSE SrcType$ = CHR$(1)
  IF DstHandle = 0 THEN DstType$ = CHR$(0) ELSE DstType$ = CHR$(1)

  'Create a buffer containing the copy information
  CopyInfo$ = MKL$(Length&) + SrcType$ + MKI$(SrcHandle) + MKI$(SrcOffset) + MKI$(SrcSegment) + DstType$ + MKI$(DstHandle) + MKI$(DstOffset) + MKI$(DstSegment)

  Regs.ax = &H5700                           'Copy the memory region
  Regs.ds = VARSEG(CopyInfo$)                'described in the buffer
  Regs.si = SADD(CopyInfo$)
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

SUB EMS.DeallocPages (Handle)

  'Deallocates the EMS pages allocated the EMS handle [Handle].
  'You MUST remember to call the sub before your program ends
  'if you allocate any memory!

  Regs.ax = &H4500                           'Release the pages allocated to [Handle]
  Regs.dx = Handle
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

FUNCTION EMS.ErrorMsg$

  'Returns a text string describing the error code in EMS.Error.

  SELECT CASE EMS.Error
    CASE &H0: Msg$ = "successful"
    CASE &H80: Msg$ = "internal error"
    CASE &H81: Msg$ = "hardware malfunction"
    CASE &H82: Msg$ = "busy -- retry later"
    CASE &H83: Msg$ = "invalid handle"
    CASE &H84: Msg$ = "undefined function requested by application"
    CASE &H85: Msg$ = "no more handles available"
    CASE &H86: Msg$ = "error in save or restore of mapping context"
    CASE &H87: Msg$ = "insufficient memory pages in system"
    CASE &H88: Msg$ = "insufficient memory pages available"
    CASE &H89: Msg$ = "zero pages requested"
    CASE &H8A: Msg$ = "invalid logical page number encountered"
    CASE &H8B: Msg$ = "invalid physical page number encountered"
    CASE &H8C: Msg$ = "page-mapping hardware state save area is full"
    CASE &H8D: Msg$ = "save of mapping context failed"
    CASE &H8E: Msg$ = "restore of mapping context failed"
    CASE &H8F: Msg$ = "undefined subfunction"
    CASE &H90: Msg$ = "undefined attribute type"
    CASE &H91: Msg$ = "feature not supported"
    CASE &H92: Msg$ = "successful, but a portion of the source region has been overwritten"
    CASE &H93: Msg$ = "length of source or destination region exceeds length of region allocated to either source or destination handle"
    CASE &H94: Msg$ = "conventional and expanded memory regions overlap"
    CASE &H95: Msg$ = "offset within logical page exceeds size of logical page"
    CASE &H96: Msg$ = "region length exceeds 1 MB"
    CASE &H97: Msg$ = "source and destination EMS regions have same handle and overlap"
    CASE &H98: Msg$ = "memory source or destination type undefined"
    CASE &H9A: Msg$ = "specified alternate map register or DMA register set not supported"
    CASE &H9B: Msg$ = "all alternate map register or DMA register sets currently allocated"
    CASE &H9C: Msg$ = "alternate map register or DMA register sets not supported"
    CASE &H9D: Msg$ = "undefined or unallocated alternate map register or DMA register set"
    CASE &H9E: Msg$ = "dedicated DMA channels not supported"
    CASE &H9F: Msg$ = "specified dedicated DMA channel not supported"
    CASE &HA0: Msg$ = "no such handle name"
    CASE &HA1: Msg$ = "a handle found had no name, or duplicate handle name"
    CASE &HA2: Msg$ = "attempted to wrap around 1M conventional address space"
    CASE &HA3: Msg$ = "source array corrupted"
    CASE &HA4: Msg$ = "operating system denied access"
    CASE ELSE: Msg$ = "undefined error: " + HEX$(EMS.Error) + "h"
  END SELECT

  EMS.ErrorMsg$ = Msg$

END FUNCTION

SUB EMS.ExchMem (Length&, SrcHandle, SrcSegment, SrcOffset, DstHandle, DstSegment, DstOffset)

  'Exhanges memory from EMS or base memory to EMS or base memory, where:
  '
  'Length&    = Length of memory to exchange in bytes
  'SrcHandle  = EMS handle of source memory (use 0 if source is base memory)
  'SrcSegment = Segment of source memory (or page number if source is EMS)
  'SrcOffset  = Offset of source memory
  'DstHandle  = EMS handle of destination memory (use 0 if destination is base memory)
  'DstSegment = Segment of destination memory (or page number if destination is EMS)
  'DstOffset  = Offset of destination memory

  'Determine the source and destination memory types by checking the handles
  IF SrcHandle = 0 THEN SrcType$ = CHR$(0) ELSE SrcType$ = CHR$(1)
  IF DstHandle = 0 THEN DstType$ = CHR$(0) ELSE DstType$ = CHR$(1)

  'Create a buffer containing the copy information
  ExchInfo$ = MKL$(Length&) + SrcType$ + MKI$(SrcHandle) + MKI$(SrcOffset) + MKI$(SrcSegment) + DstType$ + MKI$(DstHandle) + MKI$(DstOffset) + MKI$(DstSegment)

  Regs.ax = &H5701                           'Exchange the memory region
  Regs.ds = VARSEG(ExchInfo$)                'described in the buffer
  Regs.si = SADD(ExchInfo$)
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

FUNCTION EMS.FreeHandles

  'Returns the number of free (available) EMS handles.

  Regs.ax = &H4B00                             'Get the # of handles in use
  InterruptX &H67, Regs, Regs
  UsedHandles = Regs.bx

  Regs.ax = &H5402                             'Get the total # of handles
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100    'Store the status code
  TotalHandles = Regs.bx

  EMS.FreeHandles = TotalHandles - UsedHandles 'Subtract to get the # of free handles

END FUNCTION

FUNCTION EMS.FreePages

  'Returns the number of free (available) EMS pages
  '(Multiply by 16 to get the amount free EMS in KB.)

  Regs.ax = &H4200                           'Get the # of free pages
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code
  EMS.FreePages = Regs.bx

END FUNCTION

FUNCTION EMS.Init

  'Returns true (-1) if an EMM is installed
  'or false (0) if an EMM is not installed.

  Regs.ax = &H3567                        'Get the interrupt vector for int 67h
  InterruptX &H21, Regs, Regs
  DEF SEG = Regs.es                       'Point to the interrupt segment
  FOR x = 10 TO 17                        'Store the 8 bytes at ES:0A in EMM$
    EMM$ = EMM$ + CHR$(PEEK(x))
  NEXT
  IF EMM$ <> "EMMXXXX0" THEN
    EMS.Init = 0              'EMM not installed
  ELSE
    EMS.Init = -1             'EMM installed
  END IF

END FUNCTION

SUB EMS.MapPage (Physical, Logical, Handle)

  'Maps the logical EMS page [Logical] (allocated to the handle [Handle])
  'to the physical page [Physical] in the EMS page frame.

  Regs.ax = &H4400 + Physical                'Map the logical page [Logical]
  Regs.bx = Logical                          'to the physical page [Physical]
  Regs.dx = Handle
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

SUB EMS.MapXPages (PhysicalStart, LogicalStart, NumPages, Handle)

  'Maps up to 4 logical EMS pages to physical pages in the page frame, where:
  '
  'PhysicalStart = Physical page first logical page is mapped to
  'LogicalStart  = First logical page to map
  'NumPages      = Number of pages to map (1 to 4)
  'Handle        = EMS handle logical pages are allocated to

  'Create a buffer containing the page information
  FOR x = 0 TO NumPages - 1
    MapInfo$ = MapInfo$ + MKI$(LogicalStart + x) + MKI$(PhysicalStart + x)
  NEXT

  Regs.ax = &H5000                           'Map the pages in the buffer
  Regs.cx = NumPages                         'to the pageframe
  Regs.dx = Handle
  Regs.ds = VARSEG(MapInfo$)
  Regs.si = SADD(MapInfo$)
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code

END SUB

FUNCTION EMS.PageFrame

  'Returns the segment of the EMS page frame

  Regs.ax = &H4100                           'Get the segment of the page frame
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Save the status code
  EMS.PageFrame = Regs.bx

END FUNCTION

FUNCTION EMS.TotalPages

  'Returns the total number of EMS pages
  '(Multiply by 16 to get the total amount of EMS in KB.)

  Regs.ax = &H4200                           'Get the # of total pages
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Store the status code
  EMS.TotalPages = Regs.dx

END FUNCTION

FUNCTION EMS.Version$

  'Returns a string containing the EMM version.
  '(Must be "4.0" or greater to use our routines.)

  Regs.ax = &H4600                           'Get the EMM version
  InterruptX &H67, Regs, Regs
  EMS.Error = (Regs.ax AND &HFF00&) \ &H100  'Save the status code

  Version = Regs.ax AND &HFF                 'Split the version number into
  Major = (Version AND &HF0) \ &H10          'its major and minor counterparts
  Minor = Version AND &HF
  EMS.Version$ = LTRIM$(STR$(Major)) + "." + LTRIM$(STR$(Minor))

END FUNCTION

SUB SB.GetConfig

  'Reads the BLASTER environment variable and gets the Sound Blaster base
  'address and 8-bit DMA channel from it.

  Config$ = UCASE$(ENVIRON$("BLASTER"))      'Get the variable from DOS

  FOR x = 1 TO LEN(Config$)                  'Look at each character in it
    SELECT CASE MID$(Config$, x, 1)
      CASE "A"                               'We found an "A", so the next 3
					     '  characters are the base
					     '  address in hex.
	SB.BaseAddr = VAL("&H" + MID$(Config$, x + 1, 3))

      CASE "D"                               'We found a "D", so the next
					     '  character is the 8-bit DMA
					     '  channel.
	SB.DMAchan = VAL(MID$(Config$, x + 1, 1))
    END SELECT
  NEXT

END SUB

FUNCTION SB.Init

  'Initializes the Sound Blaster by resetting the DSP chip, and determines
  'which DMA registers to use based on the selected DMA channel.
  '
  'The sound card configuration must be set (either by SB.GetConfig or
  'manually) prior to calling this function.
  '
  'If the DSP is successfully reset, this function will return -1. If the
  'DSP could not be reset or the DMA channel is invalid, it will return 0.

  OUT SB.BaseAddr + &H6, 1                  'Send a "1" to the DSP reset reg
  OUT SB.BaseAddr + &H6, 0                  'Send a "0" to the DSP reset reg

  FOR ResetDSP& = 1 TO 65535                'Wait up to 65,535 reads

    IF INP(SB.BaseAddr + &HA) = &HAA THEN   'DSP read reg returned AAh,
      EXIT FOR                              '  which means it has been reset.

    ELSEIF ResetDSP& = 65535 THEN           'Still no success after 65,535
      SB.Init = 0                           '  tries, so we must have the
      EXIT FUNCTION                         '  wrong base address or there is
    END IF                                  '  no Sound Blaster card.
  NEXT

  SELECT CASE SB.DMAchan      'Since we know which DMA channel the Sound
			      '  Blaster is using, we can set up the
			      '  channel-specific registers beforehand to
			      '  save a little time.

    CASE 0                    'DMA Channel 0 (8-bit)
      SB.DMApage = &H87       'Page register = 87h
      SB.DMAadd = &H0         'Address register = 0h
      SB.DMAlen = &H1         'Length register = 1h

    CASE 1                    'DMA Channel 1 (8-bit)
      SB.DMApage = &H83       'Page register = 83h
      SB.DMAadd = &H2         'Address register = 2h
      SB.DMAlen = &H3         'Length register = 3h

    CASE 2                    'DMA Channel 2 (8-bit)
      SB.DMApage = &H81       'Page register = 81h
      SB.DMAadd = &H4         'Address register = 4h
      SB.DMAlen = &H5         'Length register = 5h

    CASE 3                    'DMA Channel 3 (8-bit)
      SB.DMApage = &H82       'Page register = 82h
      SB.DMAadd = &H6         'Address register = 6h
      SB.DMAlen = &H7         'Length register = 7h

    CASE ELSE                 'The DMA channel is either 16-bit or invalid
      SB.Init = 0             'Return error status
      EXIT FUNCTION
  END SELECT

  OUT &HA, SB.DMAchan + &H4   'Reset the DMA controller by setting the mask
			      '  bit in the DMA mask register and clearing
  OUT &HC, &H0                '  any current transfers by sending a 0 to the
			      '  DMA clear register.

  SB.Sound = 1                'Set the last playing sound to 1
  SB.Init = -1                'Return the success code

END FUNCTION

FUNCTION SB.InUse

  'Returns true (-1) if the Sound Blaster DMA channel is in use
  'or false (0) if it's not.

  OUT &HC, &H0                  'Send 0h to the DMA clear register

  'Get the number of bytes left to be transferred from the DMA length
  'register. Since registers are 8-bit, we need to read the low byte first
  'and then read the high byte.
  BytesLeft& = INP(SB.DMAlen) + INP(SB.DMAlen) * 256&

  IF BytesLeft& = &HFFFF& OR BytesLeft& = 0 THEN  'When the DMA controller is
						  '  not transferring data,
						  '  it will return either
						  '  FFFFh or 0.

    SB.InUse = 0                      'No data is being transferred
  ELSE
    SB.InUse = -1                     'Data is still being transferred
  END IF

END FUNCTION

FUNCTION SB.LoadSound (Filename$, Slot, Priority, Handle)

  'Loads a sound from a wave file into a sound "slot" in EMS, where:
  '
  'Filename$ = Filename of an 8-bit mono PCM wave file <=23 KHz. If the file
  '            is larger than 64k, only the first 64k will be loaded.
  'Slot      = Sound slot to load the sound into
  'Priority  = Priority to assign to the sound
  '            (For example, 1 = #1 priority, 2 = #2 priority, etc.)
  'Handle    = EMS handle of memory to store sounds in
  '
  'Returns true (-1) if the sound was loaded successfully
  'or false (0) if there was an error.

  WaveFile = FREEFILE
  OPEN Filename$ FOR BINARY AS #WaveFile

  IF LOF(WaveFile) = 0 THEN       'The file length is 0, so assume that
    CLOSE #WaveFile               '  we created the file when we opened it
    KILL Filename$                '  and delete it.
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  RiffID$ = SPACE$(4)
  GET #WaveFile, , RiffID$        'Check the RIFF ID string. If it's not
  IF RiffID$ <> "RIFF" THEN       '  "RIFF", then the wave file is invalid.
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , RiffLen&       'Get the RIFF length and ignore it :)

  WaveID$ = SPACE$(4)
  GET #WaveFile, , WaveID$        'Get the wave ID string. If it's not
  IF WaveID$ <> "WAVE" THEN       '  "WAVE", then the wave file is invalid.
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  FormatID$ = SPACE$(4)
  GET #WaveFile, , FormatID$      'Get the format ID string. If it's not
  IF FormatID$ <> "fmt " THEN     '  "fmt ", then the wave file is invalid.
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , FormatLen&     'Get the format length and ignore it

  GET #WaveFile, , FormatTag      'Get the format tag, which defines what
				  '  format the data is in. This needs to be
				  '  "1", which is uncompressed PCM.
  IF FormatTag <> 1 THEN          'If it's something else, we can't play it
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , NumChannels    'Get the # of channels. This needs to be "1"
				  '  (because we can only play mono sounds)
  IF NumChannels <> 1 THEN        'If it's stereo, we can't play it
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , Frequency&     'Get the sound frequency (sampling rate)
  IF Frequency& > 23000 THEN      'We can't play sounds > 23 KHz
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , TransferRate&  'Get the data transfer rate and ignore it

  GET #WaveFile, , BlockAlign     'Get the block alignment.
  IF BlockAlign <> 1 THEN         'If it's not "1", then it's not an 8-bit
				  '  wave and we can't play it.
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , ExtraData      'Get the extra data and ignore it

  DataID$ = SPACE$(4)
  GET #WaveFile, , DataID$        'Get the data ID string. If it's not
  IF DataID$ <> "data" THEN       '  "data", then the wave file is invalid.
    SB.LoadSound = 0              'Return an error code
    EXIT FUNCTION
  END IF

  GET #WaveFile, , DataLen&                  'Get the sound data length
  IF DataLen& > 65536 THEN DataLen& = 65536  'If the sound is greater than
					     '  64K, load only the first 64K


  BufferSize = 1024              'Set the buffer size to 1K. For faster
  Buffer$ = SPACE$(BufferSize)   'loading, you can increase the buffer size.
				 '(Try 4096 to 16384). However, if the
				 'buffer is too large you may run out of
				 'string space in your program.

  DataRead& = 0                  'We haven't read any data yet...

  DO
    IF DataRead& + BufferSize > DataLen& THEN  'If the buffer is larger than
					       '  the data left to read, we
      Buffer$ = SPACE$(DataLen& - DataRead&)   '  need to adjust it so we
					       '  don't read past the end
    END IF                                     '  of the file.

    GET #WaveFile, , Buffer$                   'Read the data into the buffer

    Page = DataRead& \ 16384                   'Determine the EMS page and
    Offset = DataRead& - Page * 16384&         '  offset to load the data
					       '  into.

    Page = Page + (Slot - 1) * 4               'Adjust the page depending on
					       '  which slot we're using.

    'Copy the data from the buffer to EMS
    EMS.CopyMem LEN(Buffer$), 0, VARSEG(Buffer$), SADD(Buffer$), Handle, Page, Offset

    DataRead& = DataRead& + LEN(Buffer$)       'Increase the number of bytes
					       '  read by the size of the
					       '  data buffer.

  LOOP UNTIL DataRead& = DataLen&              'Loop around until all the
					       '  data has been loaded.

  Buffer$ = ""        'Set the buffer to null to restore the string space

  CLOSE #WaveFile

  SB.SlotInfo(Slot, 1) = Frequency&    'Save the frequency,
  SB.SlotInfo(Slot, 2) = DataLen&      '  length,
  SB.SlotInfo(Slot, 3) = Priority      '  and priority of the sound.

  SB.LoadSound = -1                    'Sound loaded successfully

END FUNCTION

SUB SB.Pause

  'Pauses the sound currently playing

  SB.WriteDSP &HD0

END SUB

SUB SB.PlaySound (Slot, Handle)

  'Plays a sound from a "slot" in EMS, where:
  '
  'Slot   = Sound slot to play
  'Handle = EMS handle of memory sound is stored in

  'If a sound is already playing with a higher priority, don't interrupt it.
  IF SB.InUse AND SB.SlotInfo(SB.Sound, 3) < SB.SlotInfo(Slot, 3) THEN
    EXIT SUB
  END IF

  SoundFreq& = SB.SlotInfo(Slot, 1)             'Get the sound frequency
  SoundLen& = SB.SlotInfo(Slot, 2) - 1          'Get the sound length

  Address& = (&H10000 + EMS.PageFrame) * 16&    'Calculate the 20-bit address
  Page = (&H10000 + EMS.PageFrame) / &H1000     '  and page of the EMS page
						'  frame.

  EMS.MapXPages 0, (Slot - 1) * 4, 4, Handle    'Map the sound (64K, 4 pages)
						'  to the EMS pageframe.

  'Program the DMA controller
  OUT &HA, SB.DMAchan + &H4                     'Mask the DMA channel to use
						'  by setting the mask bit in
						'  the DMA mask register.

  OUT &HC, &H0                                  'Clear any current transfers
						'  by sending a 0 to the DMA
						'  clear register.

  OUT &HB, SB.DMAchan + &H48                    'Set the transfer mode to
						'  "output" with the DMA mode
						'   register.

  OUT SB.DMAadd, Address& AND &HFF               'Set the low and
  OUT SB.DMAadd, (Address& AND &HFF00&) \ &H100  '  high byte of the address.

  OUT SB.DMApage, Page                             'Set the page.

  OUT SB.DMAlen, SoundLen& AND &HFF                'Set the low and
  OUT SB.DMAlen, (SoundLen& AND &HFF00&) \ &H100   '  high byte of the sound
						   '  length.

  OUT &HA, SB.DMAchan       'Clear the mask bit in the DMA mask register


  'Program the DSP chip
  SB.WriteDSP &H40                         'Select the "set time transfer
					   '  constant" function.

  SB.WriteDSP 256 - 1000000 \ SoundFreq&   'Calculate and send the constant.

  SB.WriteDSP &H14                         'Select the "8-bit single-cycle
					   '  DMA output" function.

  SB.WriteDSP SoundLen& AND &HFF                 'Send the low and
  SB.WriteDSP ((SoundLen& AND &HFF00&) \ &H100)  '  high byte of the
						 '  sound length

  SB.Sound = Slot  'Save the slot number of the currently playing sound

END SUB

SUB SB.Resume

  'Resumes the sound currently playing

  SB.WriteDSP &HD4

END SUB

SUB SB.SpeakerOff

  'Turns the "speaker" off. This actually just disables the digitized sound
  'output, effectively muting the sound.

  SB.WriteDSP &HD3

END SUB

SUB SB.SpeakerOn

  'Turns the "speaker" on. This actually just enables the digitized sound
  'output so that the sound can be heard.

  SB.WriteDSP &HD1

END SUB

SUB SB.WriteDSP (Byte)

  'Writes the value [Byte] to the DSP write register.

  DO
    Ready = INP(SB.BaseAddr + &HC)  'Wait until the DSP is ready
  LOOP WHILE Ready AND &H80         '  to accept the data

  OUT SB.BaseAddr + &HC, Byte       'Send the value

END SUB

